#' @title Savings game
#' @description Given a cost game, this function returns the associated savings game.
#' @param c The characteristic function of a cost game, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{c} is introduced in binary order instead of lexicographic order.
#' @return The characteristic function of the savings game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Let \eqn{c\in G^N} be a cost game. Its associated savings game, \eqn{v_c\in G^N}, is defined by
#' \deqn{v_{c}(S)=\sum_{i\in S}c(i)-c(S) \text{ for each }S\in 2^N.}
#' Thus, for each coalition \eqn{S}, \eqn{v_{c}(S)} can be interpreted as the collective reduction of cost resulting from the cooperation of the members of \eqn{S}, with respect to the scenario of non-cooperation.
#' @seealso \link{airfieldgame}, \link{zeronormalizedgame}
#' @examples
#' savingsgame(c(360,60,288,390,468,318,468))
#' v.random <- rnorm(2^5-1,58,13)
#' savingsgame(v.random) == -zeronormalizedgame(v.random)
#' @export

savingsgame = function(c, binary = FALSE) {# Principio de la función

  nC=length(c)#Número de coaliciones.
  n=log(nC+1)/log(2)#Número de jugadores.
  if (n>floor(n))
  {
    stop("'c' must have length 2^n-1 for some n.")
  }

  return(-zeronormalizedgame(v=c,binary=binary))
}# Fin de la función
