% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TTTE_Analytical.R
\encoding{UTF-8}
\name{TTTE_Analytical}
\alias{TTTE_Analytical}
\title{Empirical Total Time on Test (TTT), analytic version.}
\usage{
TTTE_Analytical(
  formula,
  response = NULL,
  scaled = TRUE,
  data = NULL,
  method = c("Barlow", "censored"),
  partition_method = NULL,
  silent = FALSE,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{\link{formula}} with the response on
the left of an operator \code{~}. The right side can be a factor
variable as term or an \code{1} if a classification by factor
levels is not desired.}

\item{response}{an optional numeric vector with data of the response variable.
Using this argument is equivalent to define a formula with the
right side such as \code{~ 1}. See the fourth example below.}

\item{scaled}{logical. If \code{TRUE} (default value), scaled TTT is computed.}

\item{data}{an optional data frame containing the variables (response and the
factor, if it is desired). If data is not specified, the variables
are taken from the environment from which \code{TTT_analytical}
is called.}

\item{method}{a character specifying the method of computation. There are two
options available: \code{'Barlow'} and \code{'censored'}. Further
information can be found in the \strong{Details} section.}

\item{partition_method}{a list specifying cluster formation when the covariate in
\code{formula} is numeric, or when the data has several
covariates. 'quantile-based' method is the only one
currently available (See the last example).}

\item{silent}{logical. If TRUE, warnings of \code{TTTE_Analytical} are suppressed.}

\item{...}{further arguments passing to \code{\link[survival]{survfit}}.}
}
\value{
A list with class object \code{Empirical.TTT} containing a list with the
following information:
\item{i/n`}{A matrix containing the empirical quantiles. This matrix has the
number of columns equals to the number of levels of the factor
considered (number of strata).}
\item{phi_n}{A matrix containing the values of empirical TTT. his matrix has the
number of columns equals to the number of levels of the factor
considered (number of strata).}
\item{strata}{A numeric named vector storing the number of observations per strata,
and the name of each strata (names of the levels of the factor).}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

This function allows to compute the TTT curve from a formula
containing a factor type variable (classification variable).
}
\details{
When \code{method} argument is set as \code{'Barlow'}, this function
uses the original expression of empirical TTT presented by
\insertCite{Barlow1979;textual}{EstimationTools}  and used by
\insertCite{Aarset1987;textual}{EstimationTools}:

\deqn{\phi_n\left( \frac{r}{n}\right) = \frac{\left( \sum_{i=1}^{r} T_{(i)} \right) +
(n-r)T_{(r)}}{\sum_{i=1}^{n} T_i}}

where \eqn{T_{(r)}}{T_(r)} is the \eqn{r^{th}}{rth} order statistic, with
\eqn{r=1,2,\dots, n}, and \eqn{n} is the sample size. On the other hand, the option
\cite{'censored'} is an implementation based on integrals presented in
\insertCite{Westberg1994;textual}{EstimationTools}, and using
\code{\link[survival]{survfit}} to compute the Kaplan-Meier estimator:

\deqn{\phi_n\left( \frac{r}{n}\right) = \sum_{j=1}^{r} \left[ \prod_{i=1}^{j}
\left( 1 - \frac{d_i}{n_i}\right) \right] \left(T_{(j)} - T_{(j-1)} \right)}
}
\examples{
library(EstimationTools)

#--------------------------------------------------------------------------------
# Example 1: Scaled empirical TTT from 'mgus1' data from 'survival' package.

TTT_1 <- TTTE_Analytical(Surv(stop, event == 'pcm') ~1, method = 'cens',
                         data = mgus1, subset=(start == 0))
head(TTT_1$`i/n`)
head(TTT_1$phi_n)
print(TTT_1$strata)


#--------------------------------------------------------------------------------
# Example 2: Scaled empirical TTT using a factor variable with 'aml' data
# from 'survival' package.

TTT_2 <- TTTE_Analytical(Surv(time, status) ~ x, method = "cens", data = aml)
head(TTT_2$`i/n`)
head(TTT_2$phi_n)
print(TTT_2$strata)

#--------------------------------------------------------------------------------
# Example 3: Non-scaled empirical TTT without a factor (arbitrarily simulated
# data).

set.seed(911211)
y <- rweibull(n=20, shape=1, scale=pi)
TTT_3 <- TTTE_Analytical(y ~ 1, scaled = FALSE)
head(TTT_3$`i/n`)
head(TTT_3$phi_n)
print(TTT_3$strata)


#--------------------------------------------------------------------------------
# Example 4: non-scaled empirical TTT without a factor (arbitrarily simulated
# data) using the 'response' argument (this is equivalent to Third example).

set.seed(911211)
y <- rweibull(n=20, shape=1, scale=pi)
TTT_4 <- TTTE_Analytical(response = y, scaled = FALSE)
head(TTT_4$`i/n`)
head(TTT_4$phi_n)
print(TTT_4$strata)

#--------------------------------------------------------------------------------
# Eample 5: empirical TTT with a continuously variant term for the shape
# parameter in Weibull distribution.

x <- runif(50, 0, 10)
shape <- 0.1 + 0.1*x
y <- rweibull(n = 50, shape = shape, scale = pi)

partitions <- list(method='quantile-based',
                   folds=5)
TTT_5 <- TTTE_Analytical(y ~ x, partition_method = partitions)
head(TTT_5$`i/n`)
head(TTT_5$phi_n)
print(TTT_5$strata)
plot(TTT_5) # Observe changes in Empirical TTT

#--------------------------------------------------------------------------------

}
\references{
\insertRef{Barlow1979}{EstimationTools}

\insertRef{Aarset1987}{EstimationTools}

\insertRef{Klefsjo1991}{EstimationTools}

\insertRef{Westberg1994}{EstimationTools}
}
\seealso{
\code{\link{plot.EmpiricalTTT}}
}
\author{
Jaime Mosquera Gutiérrez, \email{jmosquerag@unal.edu.co}
}
\concept{EmpiricalTTT}
