% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simpA.param.R
\name{simpA.param}
\alias{simpA.param}
\title{Semiparametric testing of the simplifying assumption}
\usage{
simpA.param(
  X1,
  X2,
  X3,
  family,
  testStat = "T2c",
  typeBoot = "boot.NP",
  h,
  nBootstrap = 100,
  kernel.name = "Epanechnikov",
  truncVal = NULL,
  numericalInt = list(kind = "legendre", nGrid = 10)
)
}
\arguments{
\item{X1}{vector of \code{n} observations of the first conditioned variable}

\item{X2}{vector of \code{n} observations of the second conditioned variable}

\item{X3}{vector of \code{n} observations  of the conditioning variable}

\item{family}{the chosen family of copulas
(see the documentation of the class \code{VineCopula::\link[VineCopula]{BiCop}()}
for the available families).}

\item{testStat}{name of the test statistic to be used.
The only choice implemented yet is \code{'T2c'}.}

\item{typeBoot}{the type of bootstrap to be used.
(see Derumigny and Fermanian, 2017, p.165).
Possible values are
\itemize{
   \item \code{"boot.NP"}: usual (Efron's) non-parametric bootstrap

   \item \code{"boot.pseudoInd"}: pseudo-independent bootstrap

   \item \code{"boot.pseudoInd.sameX3"}: pseudo-independent bootstrap
   without resampling on \eqn{X_3}

   \item \code{"boot.pseudoNP"}: pseudo-non-parametric bootstrap

   \item \code{"boot.cond"}: conditional bootstrap

   \item \code{"boot.paramInd"}: parametric independent bootstrap

   \item \code{"boot.paramCond"}: parametric conditional bootstrap
}}

\item{h}{the bandwidth used for kernel smoothing}

\item{nBootstrap}{number of bootstrap replications}

\item{kernel.name}{the name of the kernel}

\item{truncVal}{the value of truncation for the integral,
i.e. the integrals are computed from \code{truncVal} to \code{1-truncVal}
instead of from 0 to 1.
Note that \code{truncVal} must be in the interval \eqn{[0, 0.5)},
i.e. \eqn{0} is allowed but not \eqn{0.5}.

The default is \code{truncVal = NULL}, which actually means that
\code{truncVal = h} if \code{h < 0.5} and \code{truncVal = 0} else.}

\item{numericalInt}{parameters to be given to
\code{statmod::\link[statmod]{gauss.quad}}, including the number of
quadrature points and the type of interpolation.}
}
\value{
a list containing
\itemize{
    \item \code{true_stat}: the value of the test statistic
    computed on the whole sample
    \item \code{vect_statB}: a vector of length \code{nBootstrap}
    containing the bootstrapped test statistics.
    \item \code{p_val}: the p-value of the test.
}
}
\description{
This function tests the ``simplifying assumption'' that a conditional
copula \deqn{C_{1,2|3}(u_1, u_2 | X_3 = x_3)} does not depend on the
value of the conditioning variable \eqn{x_3} in a semiparametric setting,
where the conditional copula is of the form
\deqn{C_{1,2|3}(u_1, u_2 | X_3 = x_3) = C_{\theta(x_3)}(u_1,u_2),}
for all \eqn{0 <= u_1, u_2 <= 1} and all \eqn{x_3}.
Here, \eqn{(C_\theta)} is a known family of copula and \eqn{\theta(x_3)}
is an unknown conditional dependence parameter.
In this setting, the simplifying assumption can be rewritten as
\strong{``\eqn{\theta(x_3)} does not depend on \eqn{x_3}, i.e. is a constant
function of \eqn{x_3}''}.
}
\examples{
# We simulate from a conditional copula
set.seed(1)
N = 500
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = -0.9 + 1.8 * pnorm(Z, mean = 5, sd = 2)
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1], mean = Z)
X2 = qnorm(simCopula[,2], mean = - Z)

result <- simpA.param(
   X1 = X1, X2 = X2, X3 = Z, family = 1,
   h = 0.03, kernel.name = "Epanechnikov", nBootstrap = 5)
print(result$p_val)
# In practice, it is recommended to use at least nBootstrap = 100
# with nBootstrap = 200 being a good choice.

\donttest{
set.seed(1)
N = 500
Z = rnorm(n = N, mean = 5, sd = 2)
conditionalTau = 0.8
simCopula = VineCopula::BiCopSim(N=N , family = 1,
    par = VineCopula::BiCopTau2Par(1 , conditionalTau ))
X1 = qnorm(simCopula[,1], mean = Z)
X2 = qnorm(simCopula[,2], mean = - Z)

result <- simpA.param(
   X1 = X1, X2 = X2, X3 = Z, family = 1,
   h = 0.08, kernel.name = "Epanechnikov", nBootstrap = 5)
print(result$p_val)
}

}
\references{
Derumigny, A., & Fermanian, J. D. (2017).
About tests of the “simplifying” assumption for conditional copulas.
Dependence Modeling, 5(1), 154-197.
\doi{10.1515/demo-2017-0011}
}
\seealso{
Other tests of the simplifying assumption:
\itemize{
  \item \code{\link{simpA.NP}} in a nonparametric setting
  \item \code{\link{simpA.kendallReg}}: test based on the constancy of
  conditional Kendall's tau

  \item the counterparts of these tests in the discrete conditioning setting:
  \code{\link{bCond.simpA.CKT}}
  (test based on conditional Kendall's tau)
  \code{\link{bCond.simpA.param}}
  (test assuming a parametric form for the conditional copula)
}
}
