# Copyrights 2023 by [Mark Overmeer <markov@cpan.org>].
#  For other contributors see ChangeLog.
# See the manual pages for details on the licensing terms.
# Pod stripped from pm file by OODoc 2.03.

package Math::Formula::Context;
use vars '$VERSION';
$VERSION = '0.10';


use warnings;
use strict;

use Log::Report 'math-formula';
use Scalar::Util qw/blessed/;

my $config;


sub new(%) { my $class = shift; (bless {}, $class)->init({@_}) }

sub _default($$$$)
{	my ($self, $name, $type, $value, $default) = @_;
	my $form
	  = ! $value         ? $type->new(undef, $default)
	  : ! blessed $value ? ($value ? Math::Formula->new($name, $value) : undef)
	  : $value->isa('Math::Formula') ? $value
	  : error __x"unexpected value for '{name}' in #{context}", name => $name, context => $self->name;
}

sub init($)
{	my ($self, $args) = @_;
	my $name   = $self->{MFC_name}   = $args->{name} or error __x"context without a name";

	my $config = $self->{MFC_config} = MF::FRAGMENT->new(config => $self, attributes => {
		name       => MF::NAME->new($name),
		mf_version => MF::STRING->new(undef, $Math::Formula::VERSION),
		version    => $args->{version} ? MF::STRING->new($args->{version}) : undef,
		created    => $self->_default(created => 'MF::DATETIME', $args->{created}, DateTime->now),
	});

	if(my $forms = $args->{formula})
	{	$self->add(ref $forms eq 'ARRAY' ? @$forms : $forms);
	}

	$self->{MFC_claims} = { };
	$self;
}

#--------------

sub name   { $_[0]->{MFC_name} }
sub config { $_[0]->{MFC_config} }

#--------------

sub add(@)
{	my $self = shift;
	unless(ref $_[0])
	{	my $name = shift;
		return $name =~ s/^#// ? $self->addFragment($name, @_) : $self->addFormula($name, @_);
	}

	foreach my $obj (@_)
	{	if(ref $obj eq 'HASH')
		{	$self->add($_, $obj->{$_}) for keys %$obj;
		}
		elsif(blessed $obj && $obj->isa('Math::Formula'))
		{	$self->{MFC_forms}{$obj->name} = $obj;
		}
		elsif(blessed $obj && $obj->isa('Math::Formula::Context'))
		{	$self->{MFC_frags}{$obj->name} = $obj;
		}
		else
		{	panic __x"formula add '{what}' not understood", what => $obj;
		}
	}

	undef;
}


sub addFormula(@)
{	my ($self, $first) = (shift, shift);

	my ($name, $form) =
	  !@_ && blessed $first && $first->isa('Math::Formula')
	? ($first->name, $first)
	: @_==1 && !ref $first && blessed $_[0] && $_[0]->isa('Math::Formula')
	? ($first, $_[0])
	: @_==1 && !ref $first && ref $_[0] eq 'ARRAY'
	? ($first, Math::Formula->new($first, @{$_[0]}))
	: @_ && !ref $first && !ref $_[0]
	? ($first, Math::Formula->new($first, @_))
	: panic __x"formula declaration '{name}' not understood", name => $first;

	$self->{MFC_forms}{$name} = $form;
}


sub formula($) { $_[0]->{MFC_forms}{$_[1]} }


sub addFragment($;$)
{	my $self = shift;
	my ($name, $fragment) = @_==2 ? @_ : ($_[0]->name, $_[0]);
	$self->{MFC_frags}{$name} = MF::FRAGMENT->new($name, $fragment);
}


sub fragment($) { $_[0]->{MFC_frags}{$_[1]} }


sub evaluate($$%)
{	my ($self, $name) = (shift, shift);

	# Wow, I am impressed!
	length $name or return $self->config;

	# silently ignore missing tags
	my $form = $self->formula($name);
	unless($form)
	{	warning __x"no formula '{name}' in {context}", name => $name, context => $self->name;
		return undef;
	}

	my $claims = $self->{MFC_claims};
	! $claims->{$name}++
		or error __x"recursion in expression '{name}' at {context}", name => $name, context => $self->name;

	my $result = $form->evaluate($self, @_);

	delete $claims->{$name};
	$result;
}


sub run($%)
{	my ($self, $expr, %args) = @_;
	my $name = delete $args{name} || join '#', (caller)[1,2];
	Math::Formula->new($name, $expr)->evaluate($self, %args);
}


sub value($@)
{	my $self = shift;
	my $result = $self->run(@_);
	$result ? $result->value : undef;
}

#--------------

1;
