use strict;
use warnings;

use Test::More;

use GDPR::IAB::TCFv2;

# use JSON;
# use DateTime;
# use DateTimeX::TO_JSON formatter => 'DateTime::Format::RFC3339';

subtest "range" => sub {
    subtest
      "should convert data to json using compact flag and 0/1 as booleans" =>
      sub {
        subtest "should convert data to json using yyyymmdd as date format" =>
          sub {
            my $consent = GDPR::IAB::TCFv2->Parse(
                'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
                json => {
                    verbose        => 0,
                    compact        => 1,
                    use_epoch      => 0,
                    boolean_values => [ 0, 1 ],
                    date_format    => '%Y%m%d',    # yyymmdd
                },
            );

            my $got      = $consent->TO_JSON();
            my $expected = _fixture_range_compact(
                created      => 20200427,
                last_updated => 20200427,
            );

            is_deeply $got, $expected, "must return the json hashref";

            done_testing;
          };

        subtest "should convert data to json using epoch date format" => sub {
            my $consent = GDPR::IAB::TCFv2->Parse(
                'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
                json => {
                    verbose        => 0,
                    compact        => 1,
                    use_epoch      => 1,
                    boolean_values => [ 0, 1 ],
                },
            );


            my $got      = $consent->TO_JSON();
            my $expected = _fixture_range_compact(
                created      => 1587946020,
                last_updated => 1587946020,
            );

            is_deeply $got, $expected, "must return the json hashref";

            done_testing;
        };

        done_testing;
      };

    subtest
      "should convert data to json using default (non-compact) and 0/1 as booleans"
      => sub {

        subtest "default non verbose, date as iso 8601" => sub {
            my $consent = GDPR::IAB::TCFv2->Parse(
                'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
                json => {
                    verbose        => 0,
                    compact        => 0,
                    use_epoch      => 0,
                    boolean_values => [ 0, 1 ],
                },
            );

            ok $consent->vendor_consent(23);

            my $got      = $consent->TO_JSON();
            my $expected = _fixture_range_default();

            is_deeply $got, $expected, "must return the json hashref";

            done_testing;
        };

        subtest "default verbose, date as iso 8601" => sub {
            my $consent = GDPR::IAB::TCFv2->Parse(
                'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
                json => {
                    verbose        => 1,
                    compact        => 0,
                    use_epoch      => 0,
                    boolean_values => [ 0, 1 ],
                },
            );

            ok $consent->vendor_consent(23);

            my $got      = $consent->TO_JSON();
            my $expected = _fixture_range_verbose();

            is_deeply $got, $expected, "must return the json hashref";

            done_testing;
        };

        done_testing;
      };

    subtest "publisher section" => sub {
        subtest "publisher section without publisher_tc" => sub {
            my $consent = GDPR::IAB::TCFv2->Parse(
                'COwAdDhOwAdDhN4ABAENAPCgAAQAAv___wAAAFP_AAp_4AI6ACACAA',
                json => {
                    verbose        => 0,
                    compact        => 1,
                    use_epoch      => 0,
                    boolean_values => [ 0, 1 ],
                },
            );

            my $got = $consent->TO_JSON;
            my $expected =
              { "publisher" => { "restrictions" => { "7" => { "32" => 1 } } }
              };

            is_deeply $got->{publisher}, $expected->{publisher},
              "must return the same publisher restriction section";

            done_testing;
        };
        subtest "publisher section with publisher_tc" => sub {
            subtest "without custom purposes" => sub {
                my $consent = GDPR::IAB::TCFv2->Parse(
                    'COwAdDhOwAdDhN4ABAENAPCgAAQAAv___wAAAFP_AAp_4AI6ACACAA.argAC0gAAAAAAAAAAAA',
                    json => {
                        verbose        => 0,
                        compact        => 1,
                        use_epoch      => 0,
                        boolean_values => [ 0, 1 ],
                    },
                );

                my $got      = $consent->TO_JSON;
                my $expected = {
                    "publisher" => {
                        "consents"             => [ 2, 4, 6, 8, 9, 10 ],
                        "legitimate_interests" => [ 2, 4, 5, 7, 10 ],
                        "custom_purposes"      => {
                            "consents"             => [],
                            "legitimate_interests" => [],
                        },
                        "restrictions" => { "7" => { "32" => 1 } }
                    }
                };

                is_deeply $got->{publisher}, $expected->{publisher},
                  "must return the same publisher restriction section";

                done_testing;
            };

            subtest "with custom purposes" => sub {
                my $consent = GDPR::IAB::TCFv2->Parse(
                    'COwAdDhOwAdDhN4ABAENAPCgAAQAAv___wAAAFP_AAp_4AI6ACACAA.YAAAAAAAAXA',
                    json => {
                        verbose        => 0,
                        compact        => 1,
                        use_epoch      => 0,
                        boolean_values => [ 0, 1 ],
                    },
                );

                my $got      = $consent->TO_JSON;
                my $expected = {
                    "publisher" => {
                        "consents"             => [],
                        "legitimate_interests" => [],
                        "custom_purposes"      => {
                            "consents"             => [ 1, 2 ],
                            "legitimate_interests" => [1],
                        },
                        "restrictions" => { "7" => { "32" => 1 } }
                    }
                };

                is_deeply $got->{publisher}, $expected->{publisher},
                  "must return the same publisher restriction section";

                done_testing;
            };

            done_testing;
        };

        done_testing;
    };

    subtest "TO_JSON method should return the same hashref " => sub {
        my $consent = GDPR::IAB::TCFv2->Parse(
            'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
            json => {
                verbose        => 0,
                compact        => 1,
                use_epoch      => 0,
                boolean_values => [ 0, 1 ],
            },
        );


        my $got1 = $consent->TO_JSON();
        my $got2 = $consent->TO_JSON();

        is_deeply $got1, $got2, "must return the same hashref";

        done_testing;
    };

    done_testing;

};

done_testing;

sub _fixture_range_compact {
    my (%extra) = @_;

    return {
        'tc_string' =>
          'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
        'consent_language' => 'EN',
        'purpose'          => {
            'consents'             => [],
            'legitimate_interests' => []
        },
        'cmp_id'                  => 3,
        'purpose_one_treatment'   => 0,
        'publisher'               => { 'restrictions' => {} },
        'special_features_opt_in' => [],
        'last_updated'            => '20200427',
        'use_non_standard_stacks' => 0,
        'policy_version'          => 2,
        'version'                 => 2,
        'is_service_specific'     => 0,
        'created'                 => '20200427',
        'consent_screen'          => 7,
        'vendor_list_version'     => 48,
        'cmp_version'             => 2,
        'publisher_country_code'  => 'AA',
        'vendor'                  => {
            'consents' => [
                23,
                42,
                126,
                127,
                128,
                587,
                613,
                626
            ],
            'legitimate_interests' => [
                24,
                44,
                129,
                130,
                131,
                591,
                614,
                628
            ]
        },
        %extra
    };
}

sub _fixture_range_default {
    my (%extra) = @_;

    return {
        'tc_string' =>
          'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
        'consent_language' => 'EN',
        'purpose'          => {
            'consents'             => {},
            'legitimate_interests' => {}
        },
        'cmp_id'                  => 3,
        'purpose_one_treatment'   => 0,
        'publisher'               => { 'restrictions' => {} },
        'special_features_opt_in' => {},
        'last_updated'            => '2020-04-27T00:07:00Z',
        'use_non_standard_stacks' => 0,
        'policy_version'          => 2,
        'version'                 => 2,
        'is_service_specific'     => 0,
        'created'                 => '2020-04-27T00:07:00Z',
        'consent_screen'          => 7,
        'vendor_list_version'     => 48,
        'cmp_version'             => 2,
        'publisher_country_code'  => 'AA',
        'vendor'                  => {
            'consents' => {
                '128' => 1,
                '613' => 1,
                '127' => 1,
                '626' => 1,
                '42'  => 1,
                '23'  => 1,
                '126' => 1,
                '587' => 1
            },
            'legitimate_interests' => {
                '628' => 1,
                '591' => 1,
                '130' => 1,
                '131' => 1,
                '24'  => 1,
                '129' => 1,
                '44'  => 1,
                '614' => 1
            }
        },
        %extra
    };
}

sub _fixture_range_verbose {
    my (%extra) = @_;

    return {
        'tc_string' =>
          'COyfVVoOyfVVoADACHENAwCAAAAAAAAAAAAAE5QBgALgAqgD8AQACSwEygJyAnSAMABgAFkAgQCDASeAmYBOgAA',
        'consent_language' => 'EN',
        'purpose'          => {
            'consents' => {
                '11' => 0,
                '21' => 0,
                '7'  => 0,
                '17' => 0,
                '2'  => 0,
                '22' => 0,
                '1'  => 0,
                '18' => 0,
                '23' => 0,
                '16' => 0,
                '13' => 0,
                '6'  => 0,
                '3'  => 0,
                '9'  => 0,
                '12' => 0,
                '20' => 0,
                '14' => 0,
                '15' => 0,
                '8'  => 0,
                '4'  => 0,
                '24' => 0,
                '19' => 0,
                '10' => 0,
                '5'  => 0
            },
            'legitimate_interests' => {
                '11' => 0,
                '21' => 0,
                '7'  => 0,
                '17' => 0,
                '2'  => 0,
                '22' => 0,
                '1'  => 0,
                '18' => 0,
                '23' => 0,
                '16' => 0,
                '13' => 0,
                '6'  => 0,
                '3'  => 0,
                '9'  => 0,
                '12' => 0,
                '20' => 0,
                '14' => 0,
                '15' => 0,
                '8'  => 0,
                '4'  => 0,
                '24' => 0,
                '19' => 0,
                '10' => 0,
                '5'  => 0
            }
        },
        'cmp_id'                  => 3,
        'purpose_one_treatment'   => 0,
        'publisher'               => { 'restrictions' => {} },
        'special_features_opt_in' => {
            '6'  => 0,
            '11' => 0,
            '3'  => 0,
            '7'  => 0,
            '9'  => 0,
            '12' => 0,
            '2'  => 0,
            '8'  => 0,
            '1'  => 0,
            '4'  => 0,
            '10' => 0,
            '5'  => 0
        },
        'last_updated'            => '2020-04-27T00:07:00Z',
        'use_non_standard_stacks' => 0,
        'policy_version'          => 2,
        'version'                 => 2,
        'is_service_specific'     => 0,
        'created'                 => '2020-04-27T00:07:00Z',
        'consent_screen'          => 7,
        'vendor_list_version'     => 48,
        'cmp_version'             => 2,
        'publisher_country_code'  => 'AA',
        'vendor'                  => {
            'consents' => {
                '559' => 0,
                '127' => 1,
                '32'  => 0,
                '443' => 0,
                '206' => 0,
                '118' => 0,
                '71'  => 0,
                '358' => 0,
                '331' => 0,
                '560' => 0,
                '580' => 0,
                '84'  => 0,
                '512' => 0,
                '437' => 0,
                '463' => 0,
                '194' => 0,
                '517' => 0,
                '458' => 0,
                '451' => 0,
                '220' => 0,
                '454' => 0,
                '31'  => 0,
                '578' => 0,
                '378' => 0,
                '325' => 0,
                '29'  => 0,
                '572' => 0,
                '350' => 0,
                '540' => 0,
                '226' => 0,
                '58'  => 0,
                '211' => 0,
                '153' => 0,
                '15'  => 0,
                '527' => 0,
                '431' => 0,
                '382' => 0,
                '337' => 0,
                '101' => 0,
                '340' => 0,
                '76'  => 0,
                '311' => 0,
                '62'  => 0,
                '571' => 0,
                '139' => 0,
                '389' => 0,
                '129' => 0,
                '548' => 0,
                '495' => 0,
                '418' => 0,
                '236' => 0,
                '218' => 0,
                '168' => 0,
                '135' => 0,
                '14'  => 0,
                '348' => 0,
                '145' => 0,
                '49'  => 0,
                '178' => 0,
                '285' => 0,
                '124' => 0,
                '234' => 0,
                '594' => 0,
                '23'  => 1,
                '388' => 0,
                '364' => 0,
                '96'  => 0,
                '486' => 0,
                '509' => 0,
                '160' => 0,
                '569' => 0,
                '367' => 0,
                '8'   => 0,
                '98'  => 0,
                '43'  => 0,
                '485' => 0,
                '391' => 0,
                '21'  => 0,
                '523' => 0,
                '288' => 0,
                '193' => 0,
                '460' => 0,
                '119' => 0,
                '586' => 0,
                '453' => 0,
                '324' => 0,
                '180' => 0,
                '244' => 0,
                '351' => 0,
                '410' => 0,
                '595' => 0,
                '246' => 0,
                '488' => 0,
                '61'  => 0,
                '430' => 0,
                '447' => 0,
                '536' => 0,
                '379' => 0,
                '415' => 0,
                '113' => 0,
                '152' => 0,
                '189' => 0,
                '452' => 0,
                '342' => 0,
                '579' => 0,
                '295' => 0,
                '480' => 0,
                '341' => 0,
                '438' => 0,
                '107' => 0,
                '535' => 0,
                '87'  => 0,
                '77'  => 0,
                '444' => 0,
                '541' => 0,
                '508' => 0,
                '221' => 0,
                '39'  => 0,
                '64'  => 0,
                '558' => 0,
                '417' => 0,
                '12'  => 0,
                '312' => 0,
                '45'  => 0,
                '507' => 0,
                '405' => 0,
                '260' => 0,
                '573' => 0,
                '237' => 0,
                '370' => 0,
                '309' => 0,
                '567' => 0,
                '1'   => 0,
                '506' => 0,
                '136' => 0,
                '116' => 0,
                '416' => 0,
                '144' => 0,
                '380' => 0,
                '100' => 0,
                '300' => 0,
                '286' => 0,
                '120' => 0,
                '381' => 0,
                '581' => 0,
                '308' => 0,
                '392' => 0,
                '254' => 0,
                '177' => 0,
                '496' => 0,
                '605' => 0,
                '373' => 0,
                '607' => 0,
                '205' => 0,
                '42'  => 1,
                '22'  => 0,
                '399' => 0,
                '235' => 0,
                '301' => 0,
                '436' => 0,
                '213' => 0,
                '94'  => 0,
                '51'  => 0,
                '456' => 0,
                '568' => 0,
                '296' => 0,
                '265' => 0,
                '493' => 0,
                '171' => 0,
                '386' => 0,
                '445' => 0,
                '200' => 0,
                '366' => 0,
                '329' => 0,
                '525' => 0,
                '27'  => 0,
                '272' => 0,
                '161' => 0,
                '582' => 0,
                '534' => 0,
                '400' => 0,
                '20'  => 0,
                '109' => 0,
                '151' => 0,
                '557' => 0,
                '468' => 0,
                '287' => 0,
                '475' => 0,
                '441' => 0,
                '78'  => 0,
                '413' => 0,
                '294' => 0,
                '349' => 0,
                '275' => 0,
                '515' => 0,
                '197' => 0,
                '138' => 0,
                '606' => 0,
                '137' => 0,
                '60'  => 0,
                '432' => 0,
                '519' => 0,
                '346' => 0,
                '17'  => 0,
                '427' => 0,
                '82'  => 0,
                '110' => 0,
                '333' => 0,
                '590' => 0,
                '323' => 0,
                '69'  => 0,
                '112' => 0,
                '545' => 0,
                '191' => 0,
                '224' => 0,
                '187' => 0,
                '588' => 0,
                '446' => 0,
                '262' => 0,
                '617' => 0,
                '79'  => 0,
                '212' => 0,
                '352' => 0,
                '126' => 1,
                '426' => 0,
                '251' => 0,
                '542' => 0,
                '369' => 0,
                '279' => 0,
                '176' => 0,
                '498' => 0,
                '483' => 0,
                '256' => 0,
                '372' => 0,
                '574' => 0,
                '170' => 0,
                '33'  => 0,
                '428' => 0,
                '7'   => 0,
                '26'  => 0,
                '227' => 0,
                '99'  => 0,
                '566' => 0,
                '526' => 0,
                '72'  => 0,
                '500' => 0,
                '264' => 0,
                '255' => 0,
                '533' => 0,
                '359' => 0,
                '182' => 0,
                '108' => 0,
                '604' => 0,
                '556' => 0,
                '462' => 0,
                '414' => 0,
                '232' => 0,
                '477' => 0,
                '225' => 0,
                '330' => 0,
                '142' => 0,
                '207' => 0,
                '263' => 0,
                '394' => 0,
                '167' => 0,
                '48'  => 0,
                '360' => 0,
                '610' => 0,
                '514' => 0,
                '513' => 0,
                '615' => 0,
                '50'  => 0,
                '476' => 0,
                '510' => 0,
                '393' => 0,
                '449' => 0,
                '293' => 0,
                '274' => 0,
                '549' => 0,
                '322' => 0,
                '469' => 0,
                '353' => 0,
                '575' => 0,
                '375' => 0,
                '128' => 1,
                '28'  => 0,
                '310' => 0,
                '40'  => 0,
                '589' => 0,
                '303' => 0,
                '192' => 0,
                '250' => 0,
                '614' => 0,
                '501' => 0,
                '215' => 0,
                '278' => 0,
                '490' => 0,
                '150' => 0,
                '130' => 0,
                '155' => 0,
                '387' => 0,
                '53'  => 0,
                '245' => 0,
                '626' => 1,
                '543' => 0,
                '267' => 0,
                '354' => 0,
                '461' => 0,
                '583' => 0,
                '257' => 0,
                '85'  => 0,
                '332' => 0,
                '9'   => 0,
                '425' => 0,
                '591' => 0,
                '34'  => 0,
                '539' => 0,
                '603' => 0,
                '90'  => 0,
                '276' => 0,
                '620' => 0,
                '565' => 0,
                '102' => 0,
                '520' => 0,
                '532' => 0,
                '16'  => 0,
                '55'  => 0,
                '233' => 0,
                '57'  => 0,
                '259' => 0,
                '368' => 0,
                '424' => 0,
                '316' => 0,
                '163' => 0,
                '395' => 0,
                '89'  => 0,
                '611' => 0,
                '175' => 0,
                '584' => 0,
                '35'  => 0,
                '11'  => 0,
                '492' => 0,
                '208' => 0,
                '347' => 0,
                '511' => 0,
                '434' => 0,
                '93'  => 0,
                '292' => 0,
                '291' => 0,
                '374' => 0,
                '114' => 0,
                '199' => 0,
                '442' => 0,
                '429' => 0,
                '73'  => 0,
                '409' => 0,
                '67'  => 0,
                '241' => 0,
                '198' => 0,
                '489' => 0,
                '585' => 0,
                '327' => 0,
                '320' => 0,
                '280' => 0,
                '273' => 0,
                '471' => 0,
                '622' => 0,
                '202' => 0,
                '249' => 0,
                '361' => 0,
                '465' => 0,
                '184' => 0,
                '24'  => 0,
                '140' => 0,
                '104' => 0,
                '131' => 0,
                '181' => 0,
                '412' => 0,
                '385' => 0,
                '502' => 0,
                '307' => 0,
                '314' => 0,
                '154' => 0,
                '355' => 0,
                '553' => 0,
                '159' => 0,
                '479' => 0,
                '326' => 0,
                '555' => 0,
                '47'  => 0,
                '619' => 0,
                '37'  => 0,
                '335' => 0,
                '270' => 0,
                '5'   => 0,
                '195' => 0,
                '621' => 0,
                '538' => 0,
                '524' => 0,
                '554' => 0,
                '552' => 0,
                '521' => 0,
                '598' => 0,
                '162' => 0,
                '433' => 0,
                '74'  => 0,
                '240' => 0,
                '334' => 0,
                '440' => 0,
                '230' => 0,
                '115' => 0,
                '299' => 0,
                '377' => 0,
                '103' => 0,
                '602' => 0,
                '201' => 0,
                '423' => 0,
                '612' => 0,
                '91'  => 0,
                '266' => 0,
                '467' => 0,
                '174' => 0,
                '474' => 0,
                '481' => 0,
                '214' => 0,
                '422' => 0,
                '564' => 0,
                '563' => 0,
                '97'  => 0,
                '41'  => 0,
                '52'  => 0,
                '302' => 0,
                '229' => 0,
                '503' => 0,
                '593' => 0,
                '68'  => 0,
                '188' => 0,
                '315' => 0,
                '402' => 0,
                '338' => 0,
                '576' => 0,
                '616' => 0,
                '222' => 0,
                '25'  => 0,
                '83'  => 0,
                '484' => 0,
                '305' => 0,
                '623' => 0,
                '544' => 0,
                '217' => 0,
                '328' => 0,
                '239' => 0,
                '122' => 0,
                '143' => 0,
                '158' => 0,
                '269' => 0,
                '281' => 0,
                '464' => 0,
                '363' => 0,
                '46'  => 0,
                '6'   => 0,
                '562' => 0,
                '36'  => 0,
                '518' => 0,
                '183' => 0,
                '497' => 0,
                '472' => 0,
                '362' => 0,
                '439' => 0,
                '317' => 0,
                '608' => 0,
                '132' => 0,
                '169' => 0,
                '411' => 0,
                '478' => 0,
                '384' => 0,
                '398' => 0,
                '546' => 0,
                '537' => 0,
                '407' => 0,
                '18'  => 0,
                '376' => 0,
                '522' => 0,
                '125' => 0,
                '599' => 0,
                '44'  => 0,
                '609' => 0,
                '587' => 1,
                '190' => 0,
                '95'  => 0,
                '298' => 0,
                '601' => 0,
                '313' => 0,
                '243' => 0,
                '231' => 0,
                '551' => 0,
                '529' => 0,
                '148' => 0,
                '343' => 0,
                '504' => 0,
                '397' => 0,
                '106' => 0,
                '157' => 0,
                '65'  => 0,
                '203' => 0,
                '261' => 0,
                '81'  => 0,
                '321' => 0,
                '459' => 0,
                '624' => 0,
                '86'  => 0,
                '284' => 0,
                '247' => 0,
                '371' => 0,
                '204' => 0,
                '165' => 0,
                '289' => 0,
                '2'   => 0,
                '435' => 0,
                '401' => 0,
                '186' => 0,
                '147' => 0,
                '339' => 0,
                '228' => 0,
                '531' => 0,
                '268' => 0,
                '345' => 0,
                '596' => 0,
                '172' => 0,
                '319' => 0,
                '223' => 0,
                '404' => 0,
                '613' => 1,
                '516' => 0,
                '282' => 0,
                '420' => 0,
                '121' => 0,
                '344' => 0,
                '487' => 0,
                '494' => 0,
                '238' => 0,
                '577' => 0,
                '253' => 0,
                '561' => 0,
                '448' => 0,
                '209' => 0,
                '216' => 0,
                '357' => 0,
                '117' => 0,
                '63'  => 0,
                '455' => 0,
                '600' => 0,
                '80'  => 0,
                '336' => 0,
                '457' => 0,
                '179' => 0,
                '383' => 0,
                '297' => 0,
                '277' => 0,
                '92'  => 0,
                '10'  => 0,
                '550' => 0,
                '505' => 0,
                '419' => 0,
                '133' => 0,
                '290' => 0,
                '625' => 0,
                '592' => 0,
                '149' => 0,
                '123' => 0,
                '304' => 0,
                '547' => 0,
                '210' => 0,
                '406' => 0,
                '258' => 0,
                '396' => 0,
                '482' => 0,
                '173' => 0,
                '530' => 0,
                '56'  => 0,
                '499' => 0,
                '66'  => 0,
                '19'  => 0,
                '54'  => 0,
                '365' => 0,
                '306' => 0,
                '70'  => 0,
                '470' => 0,
                '166' => 0,
                '88'  => 0,
                '141' => 0,
                '30'  => 0,
                '570' => 0,
                '403' => 0,
                '252' => 0,
                '466' => 0,
                '156' => 0,
                '134' => 0,
                '75'  => 0,
                '283' => 0,
                '618' => 0,
                '59'  => 0,
                '421' => 0,
                '450' => 0,
                '271' => 0,
                '491' => 0,
                '219' => 0,
                '318' => 0,
                '13'  => 0,
                '105' => 0,
                '473' => 0,
                '185' => 0,
                '3'   => 0,
                '597' => 0,
                '248' => 0,
                '390' => 0,
                '146' => 0,
                '111' => 0,
                '38'  => 0,
                '356' => 0,
                '408' => 0,
                '4'   => 0,
                '528' => 0,
                '164' => 0,
                '196' => 0,
                '242' => 0
            },
            'legitimate_interests' => {
                '559' => 0,
                '127' => 0,
                '32'  => 0,
                '443' => 0,
                '206' => 0,
                '118' => 0,
                '71'  => 0,
                '358' => 0,
                '331' => 0,
                '560' => 0,
                '580' => 0,
                '84'  => 0,
                '512' => 0,
                '437' => 0,
                '463' => 0,
                '194' => 0,
                '517' => 0,
                '458' => 0,
                '451' => 0,
                '220' => 0,
                '454' => 0,
                '31'  => 0,
                '578' => 0,
                '378' => 0,
                '325' => 0,
                '29'  => 0,
                '572' => 0,
                '350' => 0,
                '540' => 0,
                '226' => 0,
                '58'  => 0,
                '211' => 0,
                '153' => 0,
                '15'  => 0,
                '527' => 0,
                '431' => 0,
                '382' => 0,
                '337' => 0,
                '101' => 0,
                '340' => 0,
                '76'  => 0,
                '311' => 0,
                '62'  => 0,
                '571' => 0,
                '139' => 0,
                '389' => 0,
                '129' => 1,
                '548' => 0,
                '495' => 0,
                '418' => 0,
                '236' => 0,
                '218' => 0,
                '168' => 0,
                '135' => 0,
                '14'  => 0,
                '348' => 0,
                '145' => 0,
                '49'  => 0,
                '178' => 0,
                '285' => 0,
                '124' => 0,
                '627' => 0,
                '234' => 0,
                '594' => 0,
                '23'  => 0,
                '388' => 0,
                '364' => 0,
                '96'  => 0,
                '486' => 0,
                '509' => 0,
                '160' => 0,
                '569' => 0,
                '367' => 0,
                '8'   => 0,
                '98'  => 0,
                '43'  => 0,
                '485' => 0,
                '391' => 0,
                '21'  => 0,
                '523' => 0,
                '288' => 0,
                '193' => 0,
                '460' => 0,
                '119' => 0,
                '586' => 0,
                '453' => 0,
                '324' => 0,
                '180' => 0,
                '244' => 0,
                '351' => 0,
                '410' => 0,
                '595' => 0,
                '246' => 0,
                '488' => 0,
                '61'  => 0,
                '430' => 0,
                '447' => 0,
                '536' => 0,
                '379' => 0,
                '415' => 0,
                '113' => 0,
                '152' => 0,
                '189' => 0,
                '452' => 0,
                '342' => 0,
                '579' => 0,
                '295' => 0,
                '480' => 0,
                '341' => 0,
                '438' => 0,
                '107' => 0,
                '535' => 0,
                '87'  => 0,
                '77'  => 0,
                '444' => 0,
                '541' => 0,
                '508' => 0,
                '221' => 0,
                '39'  => 0,
                '64'  => 0,
                '558' => 0,
                '417' => 0,
                '12'  => 0,
                '312' => 0,
                '45'  => 0,
                '507' => 0,
                '405' => 0,
                '260' => 0,
                '573' => 0,
                '237' => 0,
                '370' => 0,
                '309' => 0,
                '567' => 0,
                '1'   => 0,
                '506' => 0,
                '136' => 0,
                '116' => 0,
                '416' => 0,
                '144' => 0,
                '380' => 0,
                '100' => 0,
                '300' => 0,
                '286' => 0,
                '120' => 0,
                '381' => 0,
                '581' => 0,
                '308' => 0,
                '392' => 0,
                '254' => 0,
                '177' => 0,
                '496' => 0,
                '605' => 0,
                '373' => 0,
                '607' => 0,
                '205' => 0,
                '42'  => 0,
                '22'  => 0,
                '399' => 0,
                '235' => 0,
                '301' => 0,
                '436' => 0,
                '213' => 0,
                '94'  => 0,
                '51'  => 0,
                '456' => 0,
                '568' => 0,
                '296' => 0,
                '265' => 0,
                '493' => 0,
                '171' => 0,
                '386' => 0,
                '445' => 0,
                '200' => 0,
                '366' => 0,
                '329' => 0,
                '525' => 0,
                '27'  => 0,
                '272' => 0,
                '161' => 0,
                '582' => 0,
                '534' => 0,
                '400' => 0,
                '20'  => 0,
                '109' => 0,
                '151' => 0,
                '557' => 0,
                '468' => 0,
                '287' => 0,
                '475' => 0,
                '441' => 0,
                '78'  => 0,
                '413' => 0,
                '294' => 0,
                '349' => 0,
                '275' => 0,
                '515' => 0,
                '197' => 0,
                '138' => 0,
                '606' => 0,
                '137' => 0,
                '60'  => 0,
                '432' => 0,
                '519' => 0,
                '346' => 0,
                '17'  => 0,
                '427' => 0,
                '82'  => 0,
                '110' => 0,
                '333' => 0,
                '590' => 0,
                '323' => 0,
                '69'  => 0,
                '112' => 0,
                '545' => 0,
                '191' => 0,
                '224' => 0,
                '187' => 0,
                '588' => 0,
                '446' => 0,
                '262' => 0,
                '617' => 0,
                '79'  => 0,
                '212' => 0,
                '352' => 0,
                '126' => 0,
                '426' => 0,
                '251' => 0,
                '542' => 0,
                '369' => 0,
                '279' => 0,
                '176' => 0,
                '498' => 0,
                '483' => 0,
                '256' => 0,
                '372' => 0,
                '574' => 0,
                '170' => 0,
                '33'  => 0,
                '428' => 0,
                '7'   => 0,
                '26'  => 0,
                '227' => 0,
                '99'  => 0,
                '566' => 0,
                '526' => 0,
                '72'  => 0,
                '500' => 0,
                '264' => 0,
                '255' => 0,
                '533' => 0,
                '359' => 0,
                '182' => 0,
                '108' => 0,
                '604' => 0,
                '556' => 0,
                '462' => 0,
                '414' => 0,
                '232' => 0,
                '477' => 0,
                '225' => 0,
                '330' => 0,
                '142' => 0,
                '207' => 0,
                '263' => 0,
                '394' => 0,
                '167' => 0,
                '48'  => 0,
                '360' => 0,
                '610' => 0,
                '514' => 0,
                '513' => 0,
                '615' => 0,
                '50'  => 0,
                '476' => 0,
                '510' => 0,
                '393' => 0,
                '449' => 0,
                '293' => 0,
                '274' => 0,
                '549' => 0,
                '322' => 0,
                '469' => 0,
                '353' => 0,
                '575' => 0,
                '375' => 0,
                '128' => 0,
                '28'  => 0,
                '310' => 0,
                '40'  => 0,
                '589' => 0,
                '303' => 0,
                '192' => 0,
                '250' => 0,
                '614' => 1,
                '501' => 0,
                '215' => 0,
                '278' => 0,
                '490' => 0,
                '150' => 0,
                '130' => 1,
                '155' => 0,
                '387' => 0,
                '53'  => 0,
                '245' => 0,
                '626' => 0,
                '543' => 0,
                '267' => 0,
                '354' => 0,
                '461' => 0,
                '583' => 0,
                '257' => 0,
                '85'  => 0,
                '332' => 0,
                '9'   => 0,
                '425' => 0,
                '591' => 1,
                '34'  => 0,
                '539' => 0,
                '603' => 0,
                '90'  => 0,
                '276' => 0,
                '620' => 0,
                '565' => 0,
                '102' => 0,
                '520' => 0,
                '532' => 0,
                '16'  => 0,
                '55'  => 0,
                '233' => 0,
                '57'  => 0,
                '259' => 0,
                '368' => 0,
                '424' => 0,
                '316' => 0,
                '163' => 0,
                '395' => 0,
                '89'  => 0,
                '611' => 0,
                '175' => 0,
                '584' => 0,
                '35'  => 0,
                '11'  => 0,
                '492' => 0,
                '208' => 0,
                '347' => 0,
                '511' => 0,
                '434' => 0,
                '93'  => 0,
                '292' => 0,
                '291' => 0,
                '374' => 0,
                '114' => 0,
                '199' => 0,
                '442' => 0,
                '429' => 0,
                '73'  => 0,
                '409' => 0,
                '67'  => 0,
                '241' => 0,
                '198' => 0,
                '489' => 0,
                '585' => 0,
                '327' => 0,
                '320' => 0,
                '280' => 0,
                '273' => 0,
                '471' => 0,
                '622' => 0,
                '202' => 0,
                '249' => 0,
                '361' => 0,
                '465' => 0,
                '184' => 0,
                '24'  => 1,
                '140' => 0,
                '104' => 0,
                '131' => 1,
                '181' => 0,
                '412' => 0,
                '385' => 0,
                '502' => 0,
                '307' => 0,
                '314' => 0,
                '154' => 0,
                '355' => 0,
                '553' => 0,
                '159' => 0,
                '479' => 0,
                '326' => 0,
                '555' => 0,
                '47'  => 0,
                '619' => 0,
                '37'  => 0,
                '335' => 0,
                '270' => 0,
                '5'   => 0,
                '195' => 0,
                '621' => 0,
                '538' => 0,
                '524' => 0,
                '554' => 0,
                '552' => 0,
                '521' => 0,
                '598' => 0,
                '162' => 0,
                '433' => 0,
                '74'  => 0,
                '240' => 0,
                '334' => 0,
                '440' => 0,
                '230' => 0,
                '115' => 0,
                '299' => 0,
                '377' => 0,
                '103' => 0,
                '602' => 0,
                '201' => 0,
                '423' => 0,
                '612' => 0,
                '91'  => 0,
                '266' => 0,
                '467' => 0,
                '174' => 0,
                '474' => 0,
                '481' => 0,
                '214' => 0,
                '422' => 0,
                '564' => 0,
                '563' => 0,
                '97'  => 0,
                '41'  => 0,
                '52'  => 0,
                '302' => 0,
                '229' => 0,
                '503' => 0,
                '593' => 0,
                '68'  => 0,
                '188' => 0,
                '315' => 0,
                '402' => 0,
                '338' => 0,
                '576' => 0,
                '616' => 0,
                '222' => 0,
                '25'  => 0,
                '83'  => 0,
                '484' => 0,
                '305' => 0,
                '623' => 0,
                '544' => 0,
                '217' => 0,
                '328' => 0,
                '239' => 0,
                '122' => 0,
                '143' => 0,
                '628' => 1,
                '158' => 0,
                '269' => 0,
                '281' => 0,
                '464' => 0,
                '363' => 0,
                '46'  => 0,
                '6'   => 0,
                '562' => 0,
                '36'  => 0,
                '518' => 0,
                '183' => 0,
                '497' => 0,
                '472' => 0,
                '362' => 0,
                '439' => 0,
                '317' => 0,
                '608' => 0,
                '132' => 0,
                '169' => 0,
                '411' => 0,
                '478' => 0,
                '384' => 0,
                '398' => 0,
                '546' => 0,
                '537' => 0,
                '407' => 0,
                '18'  => 0,
                '376' => 0,
                '522' => 0,
                '125' => 0,
                '599' => 0,
                '44'  => 1,
                '609' => 0,
                '587' => 0,
                '190' => 0,
                '95'  => 0,
                '298' => 0,
                '601' => 0,
                '313' => 0,
                '243' => 0,
                '231' => 0,
                '551' => 0,
                '529' => 0,
                '148' => 0,
                '343' => 0,
                '504' => 0,
                '397' => 0,
                '106' => 0,
                '157' => 0,
                '65'  => 0,
                '203' => 0,
                '261' => 0,
                '81'  => 0,
                '321' => 0,
                '459' => 0,
                '624' => 0,
                '86'  => 0,
                '284' => 0,
                '247' => 0,
                '371' => 0,
                '204' => 0,
                '165' => 0,
                '289' => 0,
                '2'   => 0,
                '435' => 0,
                '401' => 0,
                '186' => 0,
                '147' => 0,
                '339' => 0,
                '228' => 0,
                '531' => 0,
                '268' => 0,
                '345' => 0,
                '596' => 0,
                '172' => 0,
                '319' => 0,
                '223' => 0,
                '404' => 0,
                '613' => 0,
                '516' => 0,
                '282' => 0,
                '420' => 0,
                '121' => 0,
                '344' => 0,
                '487' => 0,
                '494' => 0,
                '238' => 0,
                '577' => 0,
                '253' => 0,
                '561' => 0,
                '448' => 0,
                '209' => 0,
                '216' => 0,
                '357' => 0,
                '117' => 0,
                '63'  => 0,
                '455' => 0,
                '600' => 0,
                '80'  => 0,
                '336' => 0,
                '457' => 0,
                '179' => 0,
                '383' => 0,
                '297' => 0,
                '277' => 0,
                '92'  => 0,
                '10'  => 0,
                '550' => 0,
                '505' => 0,
                '419' => 0,
                '133' => 0,
                '290' => 0,
                '625' => 0,
                '592' => 0,
                '149' => 0,
                '123' => 0,
                '304' => 0,
                '547' => 0,
                '210' => 0,
                '406' => 0,
                '258' => 0,
                '396' => 0,
                '482' => 0,
                '173' => 0,
                '530' => 0,
                '56'  => 0,
                '499' => 0,
                '66'  => 0,
                '19'  => 0,
                '54'  => 0,
                '365' => 0,
                '306' => 0,
                '70'  => 0,
                '470' => 0,
                '166' => 0,
                '88'  => 0,
                '141' => 0,
                '30'  => 0,
                '570' => 0,
                '403' => 0,
                '252' => 0,
                '466' => 0,
                '156' => 0,
                '134' => 0,
                '75'  => 0,
                '283' => 0,
                '618' => 0,
                '59'  => 0,
                '421' => 0,
                '450' => 0,
                '271' => 0,
                '491' => 0,
                '219' => 0,
                '318' => 0,
                '13'  => 0,
                '105' => 0,
                '473' => 0,
                '185' => 0,
                '3'   => 0,
                '597' => 0,
                '248' => 0,
                '390' => 0,
                '146' => 0,
                '111' => 0,
                '38'  => 0,
                '356' => 0,
                '408' => 0,
                '4'   => 0,
                '528' => 0,
                '164' => 0,
                '196' => 0,
                '242' => 0
            }
        },
        %extra
    };
}
